// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

//
// Slider
//
// @module     theme_adaptable/slider
// @copyright  2024 G J Barnard.
// @author     G J Barnard -
//               {@link https://moodle.org/user/profile.php?id=442195}
//               {@link https://gjbarnard.co.uk}
// @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later.
//

import $ from 'jquery';
import { flexslider } from 'theme_adaptable/flexslider';
import log from 'core/log';

/**
 * Initialise this module.
 */
export const init = () => {

    const sliderInit = () => {
        flexslider();

        // Slider.
        if ($('#main-slider').length) {
            $('#main-slider').flexslider({
                namespace: "flex-",       // New: {NEW} String: Prefix string attached to the class of every element generated by the plugin.
                selector: ".slides > li", // New: {NEW} Selector: Must match a simple pattern. '{container} > {slide}' -- Ignore pattern at your own peril
                animation: "slide",       // String: Select your animation type, "fade" or "slide".
                easing: "swing",          // {NEW} String: Determines the easing method used in jQuery transitions. jQuery easing plugin is supported!
                direction: "horizontal",  // String: Select the sliding direction, "horizontal" or "vertical".
                reverse: false,           // {NEW} Boolean: Reverse the animation direction.
                animationLoop: true,      // Boolean: Should the animation loop? If false, directionNav will received "disable" classes at either end.
                smoothHeight: false,      // {NEW} Boolean: Allow height of the slider to animate smoothly in horizontal mode.
                startAt: 0,               // Integer: The slide that the slider should start on. Array notation (0 = first slide).
                slideshow: true,          // Boolean: Animate slider automatically.
                slideshowSpeed: 7000,     // Integer: Set the speed of the slideshow cycling, in milliseconds.
                animationSpeed: 600,      // Integer: Set the speed of animations, in milliseconds.
                initDelay: 0,             // {NEW} Integer: Set an initialization delay, in milliseconds.
                randomize: false,         // Boolean: Randomize slide order.

                // Usability features.
                pauseOnAction: true,      // Boolean: Pause the slideshow when interacting with control elements, highly recommended.
                pauseOnHover: false,      // Boolean: Pause the slideshow when hovering over slider, then resume when no longer hovering.
                useCSS: true,             // {NEW} Boolean: Slider will use CSS3 transitions if available.
                touch: true,              // {NEW} Boolean: Allow touch swipe navigation of the slider on touch-enabled devices.
                video: false,             // {NEW} Boolean: If using video in the slider, will prevent CSS3 3D Transforms to avoid graphical glitches.

                // Primary Controls.
                controlNav: true,         // Boolean: Create navigation for paging control of each slide? Note! Leave true for manualControls usage.
                directionNav: true,       // Boolean: Create navigation for previous / next navigation? (true/false).
                prevText: "Previous",     // String: Set the text for the "previous" directionNav item.
                nextText: "Next",         // String: Set the text for the "next" directionNav item.

                // Secondary Navigation.
                keyboard: true,           // Boolean: Allow slider navigating via keyboard left/right keys.
                multipleKeyboard: false,  // {NEW} Boolean: Allow keyboard navigation to affect multiple sliders. Default behavior cuts out keyboard navigation with more than one slider present.
                mousewheel: false,        // {UPDATED} Boolean: Requires jquery.mousewheel.js (https://github.com/brandonaaron/jquery-mousewheel) - Allows slider navigating via mousewheel.
                pausePlay: false,         // Boolean: Create pause/play dynamic element.
                pauseText: 'Pause',       // String: Set the text for the "pause" pausePlay item.
                playText: 'Play',         // String: Set the text for the "play" pausePlay item.

                // Special properties.
                controlsContainer: "",    // {UPDATED} Selector: USE CLASS SELECTOR. Declare which container the navigation elements should be appended too. Default container is the FlexSlider element. Example use would be ".flexslider-container". Property is ignored if given element is not found.
                manualControls: "",       // Selector: Declare custom control navigation. Examples would be ".flex-control-nav li" or "#tabs-nav li img", etc. The number of elements in your controlNav should match the number of slides/tabs.
                sync: "",                 // {NEW} Selector: Mirror the actions performed on this slider with another slider. Use with care.
                asNavFor: "",             // {NEW} Selector: Internal property exposed for turning the slider into a thumbnail navigation for another slider.
            });

            if ($('.container.slidewrap').length) {
                $(".container.slidewrap").on('transitionend', function () {
                    var slider1 = $('#main-slider').data('flexslider');
                    slider1.resize();
                });
            }
        }

    };

    if (document.readyState !== 'loading') {
        log.debug("Adaptable ES6 Slider JS DOM content already loaded");
        sliderInit();
    } else {
        log.debug("Adaptable ES6 Slider JS DOM content not loaded");
        document.addEventListener('DOMContentLoaded', function () {
            log.debug("Adaptable ES6 Slider JS DOM content loaded");
            sliderInit();
        });
    }
};
